// background.js (MV3 service worker, ES module)
import { CONFIG, apiUrl, DEFAULT_WHITELIST } from "./config.js";
import { decryptProxyList } from "./lib/crypto.js";


const BGCONFIG = {
  LIST_URL: apiUrl(CONFIG.API.PROXY_LIST),
  CHECK_URL: apiUrl(CONFIG.API.HEALTH),
  AES_KEY_B64: "ea85c7ced76d68c465c7d1f9a26a9c04414caf61bc917dcbeac1217d2752b7bd",
  CHECK_INTERVAL_MINUTES: 5,
  CONNECT_TIMEOUT_MS: 2500,
};

let isRefreshing = false;
let cachedAuth = { username: null, email: null, token: null, vip: null };

async function debugLog(action, detail = "") {
  const { logs = [] } = await chrome.storage.local.get("logs");
  const entry = `${new Date().toLocaleTimeString()} [${action}] ${JSON.stringify(detail)}`;
  console.log(entry); // 实时查看
  await chrome.storage.local.set({ logs: [entry, ...logs].slice(0, 100) }); // 保存前100条
}

async function initializePlugin() {
  try {
    // 1. 紧急加载凭据
    await loadAuth();
    await setState({ enabled: false });

    // 2. 暴力重置代理配置，防止上次崩溃导致的残留配置卡死内核
    await chrome.proxy.settings.clear({ scope: "regular" });

    // 3. 获取状态并恢复
    const state = await getState();
    //console.log(state);
    if (state.enabled && (await isLoggedIn())) {
      // 增加微小延迟，避开浏览器启动时的资源争夺高峰
      setTimeout(() => {
        applyProxyFromState().catch(e => console.error("Startup Apply Error:", e));
      }, 300);
    }
  } catch (err) {
    console.error("Critical Init Error:", err);
  }
}
// self.addEventListener('activate', function (event) {
//   initializePlugin();
// });
// 监听浏览器启动
chrome.runtime.onStartup.addListener(initializePlugin);
// 监听插件安装或更新
chrome.runtime.onInstalled.addListener(initializePlugin);

self.addEventListener('unhandledrejection', (event) => {
  debugLog('undefind',  event.reason)
  console.error('检测到未捕获的 Promise 异常:', event.reason);
  // 可以在这里通过 chrome.storage 记录错误日志，方便你下次查看
});


async function loadAuth() {
  const { vip, email, username, token, loggedIn } = await chrome.storage.local.get([
    "vip",
    "email",
    "username",
    "token",
    "loggedIn",
  ]);
  if (loggedIn && email && username && token) cachedAuth = { username,email, token, vip };
  else cachedAuth = { username: null, email, token: null, vip: null };
}


const DEFAULT_PROXIES = [
  { id: "default-1", name: "Default Proxy", scheme: "https", host: "1.2.3.4", port: 443 },
];

async function getState() {
  const data = await chrome.storage.local.get({
    enabled: false,
    selectedId: DEFAULT_PROXIES[0].id,
    proxies: DEFAULT_PROXIES,
    health: {}, // { [id]: { alive: boolean, rtt: number, last: number } }
    userWhitelist: [], // 用户白名单：一行一个域名或 *.domain.com
  });
  return data;
}

async function setState(patch) {
  await chrome.storage.local.set(patch);
}

/** ---------------------------
 *  白名单 + PAC 相关
 * --------------------------- */

function normalizeList(list) {
  const out = [];
  const seen = new Set();
  for (const raw of list || []) {
    const s = String(raw ?? "").trim();
    if (!s) continue;
    if (s.startsWith("#")) continue;
    const v = s.toLowerCase();
    if (!seen.has(v)) {
      seen.add(v);
      out.push(v);
    }
  }
  return out;
}
function pacTokenForScheme(scheme) {
  switch ((scheme || "").toLowerCase()) {
    case "https":
      return "HTTPS";
    case "http":
      return "PROXY";
    case "socks5":
      return "SOCKS5";
    case "socks":
      return "SOCKS";
    default:
      return "PROXY";
  }
}
// 生成 PAC 脚本：命中白名单 -> PROXY，否则 DIRECT
function buildPacScript(proxy, patterns) {
  const patternsJson = JSON.stringify(patterns);
  const token = pacTokenForScheme(proxy.scheme);
  // 注意：PAC 里的 JS 运行环境是独立的，不能引用外部变量
  return `
function FindProxyForURL(url, host) {
  host = (host || "").toLowerCase();

  if (isPlainHostName(host)) return "DIRECT";
  if (host === "localhost") return "DIRECT";

  var patterns = ${patternsJson};

  function matchHost(h, p) {
    if (!p) return false;
    if (p.indexOf("*.") === 0) {
      var base = p.slice(2);
      if (h === base) return true;
      return h.length > base.length && h.endsWith("." + base);
    }
    return h === p;
  }

  for (var i = 0; i < patterns.length; i++) {
    if (matchHost(host, patterns[i])) {
      //return "PROXY ${proxy.host}:${proxy.port}; DIRECT";
      return "${token} ${proxy.host}:${proxy.port}; DIRECT";
    }
  }
  return "DIRECT";
}
`.trim();
}
function toAsciiDomain(domain) {
  try {
    // URL 会自动把中文域名转成 punycode（ASCII）
    return new URL("http://" + domain).hostname;
  } catch {
    return domain; // 解析失败就原样返回（后面还会过滤）
  }
}

function toAsciiRule(rule) {
  rule = String(rule ?? "").trim().toLowerCase();

  // 去掉常见全角符号/空格
  rule = rule.replace(/[，。；、\s]+/g, "").trim();

  if (!rule) return "";

  if (rule.startsWith("*.")) {
    const base = rule.slice(2);
    const asciiBase = toAsciiDomain(base);
    return asciiBase ? "*." + asciiBase : "";
  }

  return toAsciiDomain(rule);
}

function isAsciiOnly(s) {
  return /^[\x00-\x7F]*$/.test(s);
}


async function applyPacProxyWithWhitelist(proxy) {
  try {
    const { userWhitelist } = await getState();
    // 过滤非 ASCII 字符，防止 Chrome 底层代理内核崩溃
    const merged = normalizeList([...(DEFAULT_WHITELIST || []), ...(userWhitelist || [])])
      .map(toAsciiRule)
      .filter(Boolean)
      .filter(isAsciiOnly);
      
    const pac = buildPacScript(proxy, merged);

    // 最后的安全检查：确保 PAC 脚本字符串本身也是纯 ASCII
    if (!isAsciiOnly(pac)) {
      throw new Error("PAC 脚本包含非法字符");
    }

    await chrome.proxy.settings.set({
      value: { mode: "pac_script", pacScript: { data: pac } },
      scope: "regular",
    });
  } catch (e) {
    console.error("[重要] 代理应用失败，已回退:", e);
    // 发生错误时，将代理重置为直连，防止插件被 Chrome 判定为故障
    await clearProxy(); 
  }
}

/** ---------------------------
 *  代理列表拉取 & 健康检测（保留你原逻辑）
 * --------------------------- */

// async function fetchProxyList() {
//   const res = await fetch(BGCONFIG.LIST_URL, { cache: "no-store" });
//   if (!res.ok) throw new Error("fetch failed");

//   const j = await res.json();
//   const base64Blob = j.data;

//   const list = await decryptProxyList(base64Blob, BGCONFIG.AES_KEY_B64);
//   return list;
// }
async function fetchProxyList() {
  const res = await fetch(BGCONFIG.LIST_URL, { cache: "no-store" });
  if (!res.ok) throw new Error("fetch failed");

  const j = await res.json();
  // 解密获取完整原始列表
  const fullList = await decryptProxyList(j.data, BGCONFIG.AES_KEY_B64);
  
  // 1. 获取当前登录用户的 VIP 等级
  const { vip } = await chrome.storage.local.get({ vip: 0 });
  const userVip = Number(vip || 0);

  // 2. 应用你的特殊过滤逻辑
  const filteredList = fullList.filter(p => {
    const nodeVip = Number(p.vip || 0);
    
    // 情况 A：VIP 0 用户只能看到 VIP 0 节点
    if (userVip === 0) {
      return nodeVip === 0;
    }
    
    // 情况 B：付费用户 (VIP > 0) 看不到 VIP 0，但能看到所有 <= 自己等级的节点
    return nodeVip > 0 && userVip >= nodeVip;
  });

  // 3. 返回过滤后的列表，并确保端口为整数类型
  return filteredList.map(p => ({
    ...p,
    port: parseInt(p.port, 10)
  }));
}

// 仅用于健康检测：临时把所有流量强制走某个代理（不受白名单影响）
async function setProxyFixedForCheck(p) {
  await chrome.proxy.settings.set({
    value: {
      mode: "fixed_servers",
      rules: {
        singleProxy: { scheme: p.scheme, host: p.host, port: p.port },
        bypassList: ["<local>"],
      },
    },
    scope: "regular",
  });
}

async function clearProxy() {
  await chrome.proxy.settings.set({ value: { mode: "direct" }, scope: "regular" });
}

async function fetchWithTimeout(url, timeoutMs) {
  const controller = new AbortController();
  const t = setTimeout(() => controller.abort(), timeoutMs);
  try {
    const res = await fetch(url, { cache: "no-store", signal: controller.signal });
    return res.ok;
  } finally {
    clearTimeout(t);
  }
}

async function checkProxyByFetch(p, timeoutMs = 3000) {

  const start = Date.now();
  // 使用 Promise.race 确保绝对超时，防止 fetch 信号挂起
  return Promise.race([
    (async () => {
      try {
        //console.log(p);
        await setProxyFixedForCheck(p);
        //console.log(BGCONFIG.CHECK_URL);
        const ok = await fetchWithTimeout(BGCONFIG.CHECK_URL, timeoutMs);
        return { alive: ok, rtt: Date.now() - start };
      } catch {
        return { alive: false, rtt: Date.now() - start };
      } finally {
        await clearProxy();
      }
    })(),
    new Promise(resolve => setTimeout(() => resolve({ alive: false, rtt: timeoutMs }), timeoutMs + 500))
  ]);
}
function isValidProxy(p) {
  // 简单的正则检查：必须有 host 和 port，且 host 不能是 undefined
  if (!p.host || p.host === "undefined" || !p.port) return false;
  // 检查是否包含非法字符
  const hostRegex = /^[a-zA-Z0-9.-]+$/;
  return hostRegex.test(p.host);
}

// background.js

async function checkAllHealth() {
  // 1. 检测前必须强制同步最新的认证信息和 VIP 等级
  await loadAuth(); 

  const { proxies, health, failStats = {}, vip } = await chrome.storage.local.get({
    proxies: [],
    health: {},
    failStats: {},
    vip: 0 // 默认为 VIP 0
  });

  const userVip = Number(vip || 0);
  const newHealth = { ...health };
  const newFailStats = { ...failStats };

  // 2. 应用 VIP 阶梯过滤逻辑
  // VIP 0 只能看 0；VIP > 0 看不到 0，但能看到所有 <= 自己等级的付费节点
  const accessibleProxies = proxies.filter(p => {
    const nodeVip = Number(p.vip || 0);
    if (userVip === 0) return nodeVip === 0;
    return nodeVip > 0 && userVip >= nodeVip;
  });

  // 3. 仅检测经过过滤后的可用节点
  for (const p of accessibleProxies.slice(0, 20)) { 
    if (!isValidProxy(p)) {
      console.warn(`跳过非法地址节点: ${p.id}`);
      continue; 
    }

    // 如果连续失败次数过多则跳过，除非是手动刷新触发的检测
    if ((newFailStats[p.id] || 0) >= 3) continue; 

    try {
      // 增加 Promise.race 保护，防止 checkProxyByFetch 内部卡死
      const result = await Promise.race([
        checkProxyByFetch(p, 3000),
        new Promise(r => setTimeout(() => r({ alive: false, rtt: 3500 }), 4500))
      ]);
      
      newHealth[p.id] = { 
        alive: result.alive, 
        rtt: result.rtt, 
        last: Date.now() 
      };

      // 4. 如果检测成功，立即重置失败计数
      if (result.alive) {
        newFailStats[p.id] = 0;
      } else {
        newFailStats[p.id] = (newFailStats[p.id] || 0) + 1;
      }
    } catch (e) {
      newFailStats[p.id] = (newFailStats[p.id] || 0) + 1;
    }
  }

  // 5. 保存更新后的健康状态和统计
  await chrome.storage.local.set({ health: newHealth, failStats: newFailStats });

  // 6. 检测完成后，重新应用一次当前选择的代理，确保 PAC 脚本保持最新
  await applyProxyFromState();
}

/** ---------------------------
 *  应用代理（这里合并“白名单走代理”）
 * --------------------------- */

async function applyProxyFromState() {
  const { enabled, selectedId, proxies } = await getState();

  if (!enabled) {
    await chrome.proxy.settings.set({ value: { mode: "direct" }, scope: "regular" });
    return;
  }
  // 增加权限检查
  const details = await new Promise(r => chrome.proxy.settings.get({}, r));
  if (details.levelOfControl === 'controlled_by_other_extensions') {
    await debugLog("CONFLICT", "代理被占用，取消应用");
    return; // 优雅退出，不强制覆盖
  }

  const p = proxies.find((x) => x.id === selectedId) || proxies[0];
  if (!p) return;

  // ✅ 开启时：使用 PAC + 白名单
  await applyPacProxyWithWhitelist(p);
}
async function refreshListAndHealth() {
  // 如果正在刷新中，直接拦截，防止并发
  if (isRefreshing) {
    console.log("刷新任务正在进行中，跳过本次请求");
    return;
  }

  isRefreshing = true; // 上锁

  try {
    const list = await fetchProxyList();
    await setState({ proxies: list }); 

    const { selectedId } = await getState();
    if (!list.some((p) => p.id === selectedId)) {
      await setState({ selectedId: list[0]?.id });
    }

    // 通知 UI 列表已更新
    chrome.runtime.sendMessage({ type: "PROXY_LIST_UPDATED" }).catch(() => {});

    // 后台检测，依然不建议在 refresh 函数里 await 它，避免阻塞消息回执
    checkAllHealth().finally(() => {
      isRefreshing = false; // 检测彻底完成后解锁
    });

  } catch (e) {
    console.error("更新列表失败:", e);
    isRefreshing = false; // 报错也要记得解锁，否则插件重启前无法再次刷新
  }
}

async function isLoggedIn() {
  const { loggedIn, email, username, token, vip } = await chrome.storage.local.get(["loggedIn", "email", "username", "token", "vip"]);
  return !!(loggedIn && email && username && token && vip);
}
/** ---------------------------
 *  消息处理（合并你的 AUTH_UPDATED + 原 popup 消息 + 新 REFRESH_PAC）
 * --------------------------- */
async function handleMessage(msg) {
  await debugLog("RECV_MSG", msg.type);
  // 1. 状态查询类消息（高频，需快速响应）
  if (msg?.type === "GET_PROXY_STATUS") {
    try {
      const r = await chrome.proxy.settings.get({ incognito: false });
      return {
        ok: true,
        mode: r.value?.mode,
        hasPac: !!r.value?.pacScript?.data,
        pacHead: r.value?.pacScript?.data?.slice(0, 300) || "",
      };
    } catch (e) {
      return { ok: false, error: "获取代理状态失败" };
    }
  }

  // 2. 业务逻辑类消息
  try {
    switch (msg?.type) {
      case "AUTH_UPDATED":
        await loadAuth();
        if (await isLoggedIn()) {
          // 【关键】不要 await 这个沉重的函数，否则会导致 Service Worker 处理消息超时
          // 让它在后台静默执行，执行完后会自动通过 applyProxyFromState 更新代理
          //await chrome.storage.local.set({ failStats: {} });
          await chrome.storage.local.set({ health: {}, failStats: {} });
          refreshListAndHealth().catch(err => console.error("静默刷新失败:", err));
          return { ok: true, message: "登录更新成功，正在后台同步列表..." };
        } else {
          await setState({ enabled: false });
          await clearProxy();
          return { ok: true, message: "已退出登录并清理代理" };
        }

      case "REFRESH":
        // 同样，手动刷新也不要阻塞 Popup 的响应
        await chrome.storage.local.set({ failStats: {} });
        refreshListAndHealth().catch(err => console.error("刷新列表失败:", err));
        return { ok: true, message: "开始更新列表..." };

      case "TOGGLE":
        await setState({ enabled: !!msg.enabled });
        await applyProxyFromState();
        return { ok: true };

      case "SELECT":
        await setState({ selectedId: msg.selectedId });
        await applyProxyFromState();
        return { ok: true };

      case "CHECK":
        // 健康检测通常很慢，不阻塞响应
        checkAllHealth().catch(err => console.error("健康检测失败:", err));
        return { ok: true, message: "正在检测节点延迟..." };

      case "REFRESH_PAC":
        // 白名单更新后立即重新应用
        await applyProxyFromState();
        return { ok: true };

      default:
        return { ok: false, error: "未知的消息类型: " + msg?.type };
    }
  } catch (e) {
    console.error("handleMessage 内部捕获异常:", e);
    return { ok: false, error: String(e?.message || e) };
  }
}



chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  // 直接调用 handleMessage 并将结果传回，不进行二次包裹
  handleMessage(msg)
    .then(result => {
      sendResponse(result || { ok: true });
    })
    .catch(err => {
      sendResponse({ ok: false, error: err.message });
    });
  return true; // 保持通道开启
});

/** ---------------------------
 *  storage 变化自动刷新 PAC（白名单/代理选择变了）
 * --------------------------- */
chrome.storage.onChanged.addListener((changes, area) => {
  if (area !== "local") return;

  // 白名单变了、启用状态变了、选择变了、代理列表变了，都重建 PAC
  if (
    changes.userWhitelist ||
    changes.enabled ||
    changes.selectedId ||
    changes.proxies
  ) {
    applyProxyFromState().catch(console.error);
  }

  // auth 相关也可顺带刷新缓存
  if (changes.username || changes.token || changes.loggedIn) {
    loadAuth().catch(console.error);
  }
});

/** ---------------------------
 *  安装时初始化 & 定时检测（保留）
 * --------------------------- */

/*
chrome.runtime.onInstalled.addListener(async () => {
  const st = await getState();
  if (!st.proxies?.length) {
    await setState({ proxies: DEFAULT_PROXIES, selectedId: DEFAULT_PROXIES[0].id });
  }
  await chrome.alarms.create("healthCheck", { periodInMinutes: BGCONFIG.CHECK_INTERVAL_MINUTES });
  if (await isLoggedIn()) {
    await refreshListAndHealth();
  }
});
*/

/** ---------------------------
 *  代理鉴权（保留你原逻辑）
 * --------------------------- */
chrome.webRequest.onAuthRequired.addListener(
  (details) => {
    //console.log('vip:',cachedAuth);
    // 只处理代理认证，避免处理站点认证
    if (!details.isProxy) return {};
    //const user = cachedAuth?.username || 'wang';
    const pass = cachedAuth?.token || '';
    const email = cachedAuth?.email || '';

    const username = (cachedAuth?.vip == 0) ? 'wang' : email.replace(/[@.]/g, '_');
    debugLog("AUTH PROXY", [username, pass]);
    if (!pass) {
      forceLogout();
      return { cancel: true };
    }
    
    //if (!cachedAuth?.username || !cachedAuth?.token) return {};

    return {
      authCredentials: {
        username: username,
        password: pass,
      },
    };
  },
  { urls: ["<all_urls>"] },
  ["blocking"]
);

async function forceLogout() {
  // 清理所有登录相关的存储
  await chrome.storage.local.remove(["loggedIn", "username", "token", "vip", "email"]);
  // 重置插件状态为关闭
  await clearProxy();
  // 通知全局：认证已更新（变为未登录状态）
  chrome.runtime.sendMessage({ type: "AUTH_UPDATED" }).catch(() => {});
}
/*
 chrome.storage.local.get("logs", (data) => {
  console.log(data);
   if (data.logs && data.logs.length > 0) {
     console.log("--- 崩溃前的黑匣子日志 ---");
     console.table(data.logs);
   } else {
     console.log("未发现持久化日志，请确认 background.js 中是否已加入 debugLog 逻辑。");
   }
 });
*/